// NSF music file emulator

// Game_Music_Emu 0.4.0
#ifndef NSF_EMU_H
#define NSF_EMU_H

#include "Classic_Emu.h"
#include "Nes_Apu.h"
#include "Nes_Cpu.h"

typedef Nes_Emu Nsf_Emu;

class Nes_Emu : private Nes_Cpu, public Classic_Emu {
public:
	// Set internal gain, where 1.0 results in almost no clamping. Default gain
	// roughly matches volume of other emulators.
	Nes_Emu( double gain = 1.4 );
	
	// NSF file header
	struct header_t
	{
		char tag [5];
		byte vers;
		byte track_count;
		byte first_track;
		byte load_addr [2];
		byte init_addr [2];
		byte play_addr [2];
		char game [32];
		char author [32];
		char copyright [32];
		byte ntsc_speed [2];
		byte banks [8];
		byte pal_speed [2];
		byte speed_flags;
		byte chip_flags;
		byte unused [4];
	};
	BOOST_STATIC_ASSERT( sizeof (header_t) == 0x80 );
	
	// Load NSF data
	blargg_err_t load( Data_Reader& );
	
	// Load NSF using already-loaded header and remaining data
	blargg_err_t load( header_t const&, Data_Reader& );
	
	// Header for currently loaded NSF
	header_t const& header() const { return header_; }
	
	// Equalizer profiles for US NES and Japanese Famicom
	static equalizer_t const nes_eq;
	static equalizer_t const famicom_eq;
	
	// Read track info without loading into emulator
	static blargg_err_t read_info( Data_Reader& in, track_info_t* out, int track = 0 );

public:
	~Nes_Emu();
	Nes_Apu* apu_() { return &apu; }
	const char** voice_names() const;
	Music_Emu::track_info;
	blargg_err_t track_info( track_info_t*, int track ) const;
protected:
	void start_track_( int );
	void set_voice( int, Blip_Buffer*, Blip_Buffer*, Blip_Buffer* );
	void update_eq( blip_eq_t const& );
	blip_time_t run_clocks( blip_time_t, bool* );
	void unload();
protected:
	// initial state
	enum { bank_count = 8 };
	byte initial_banks [bank_count];
	int initial_pcm_dac;
	double gain;
	bool needs_long_frames;
	bool pal_only;
	unsigned init_addr;
	unsigned play_addr;
	int exp_flags;
	
	// timing
	Nes_Cpu::registers_t saved_state;
	nes_time_t next_play;
	long play_period;
	int play_extra;
	int play_ready;
	nes_time_t clock() const;
	nes_time_t next_irq( nes_time_t end_time );
	static void irq_changed( void* );
	
	// rom
	int total_banks;
	blargg_vector<byte> rom;
	
	blargg_err_t init_sound();
	class Nes_Namco_Apu* namco;
	class Nes_Vrc6_Apu* vrc6;
	class Nes_Fme7_Apu* fme7;
	
	// large objects
	
	header_t header_;
	
	enum { master_clock_divisor = 12 };
	enum { page_size = 0x1000 };
	enum { rom_begin = 0x8000 };
	enum { bank_select_addr = 0x5FF8 };
	enum { badop_addr = bank_select_addr };
	
	// CPU
	typedef Nes_Cpu cpu;
	void cpu_jsr( nes_addr_t );
	int cpu_read( nes_addr_t );
	void cpu_write( nes_addr_t, int );
	void cpu_write_misc( nes_addr_t, int );
	nes_addr_t handle_badop( nes_addr_t pc );
	friend class Nes_Cpu;
	
	Nes_Apu apu;
	static int pcm_read( void*, nes_addr_t );
	
	byte sram [0x2000];
	byte unmapped_code [Nes_Cpu::page_size];
};

#endif
